capture log close
clear all
macro drop _all
cd "/Volumes/Mike2025/MC_CLIENTS/2025/AER 2025/Stata/Time Varying Models-smpl2020"
set more off
set maxiter 3000

**** SET PERIOD ***
local perd "long"      // can be "long" or "short"

log using "Output/log/SFA-BC95-JTT-HN/anOpexReg24-BC95-JTT-HN-`perd'", ///
	replace text

/*--------------------------------------------------------------------------
Project: Quantonomics DNSP Opex Cost Function -  SFA Time-Varying Models
		 BC95-JTT-HN Model
Author:  Alice Giovani & Michael Cunningham
File:    anOpexReg24-BC95-JTT-HN.do
Initial Date:    19-Mar-2025
Revised Date:	 1-Aug-2025

--------------------------------------------------------------------------*/

* Period & dependent variable
if "`perd'" == "long" {
	local startyr 2006
	}
else if "`perd'" == "short" {
	local startyr 2012
}
local endyr   2020

* 
scalar define ppp_aust   		   = 1.527404363
scalar define ppp_nz    		   = 1.482877481
scalar define ppp_canada		   = 1.249076963

scalar define propex_rebase_aust   = 1.238501756
scalar define propex_rebase_nz     = 1.219668445
scalar define propex_rebase_canada = 1.160126855  

* Read data
use "Input/DNSPopex24", clear
xtset eiid year

gen int dnsp = eiid - 1000 if country == 1
qui tabulate dnsp, gen(d)
forvalues k = 1/13 {
	replace d`k' = 0 if d`k' == .
}

tabulate country, gen(jur)
correlate custnum circlen rmdem shareugc
tabulate year

summarize d* jur*

* Sample selection
*-------------------------
drop if country == 3 & year < (`startyr' - 1)
drop if country < 3 & year < `startyr'
drop if country == 3 & year > (`endyr' - 1)
drop if country < 3 & year > `endyr'

* Modify & create variables
*--------------------------

* Adressing change of VIC's DNSP from calendar to financial year in 2021
generate yr=year
replace yr=yr+0.5 if dnsp==3 & year<2021
replace yr=yr+0.5 if dnsp==8 & year<2021
replace yr=yr+0.5 if dnsp==9 & year<2021
replace yr=yr+0.5 if dnsp==11 & year<2021
replace yr=yr+0.5 if dnsp==13 & year<2021
replace yr=yr-0.25 if country==2
replace yr=yr+0.5 if country==3

replace propex = propex / propex_rebase_aust    if country==1
replace propex = propex / propex_rebase_nz      if country==2
replace propex = propex / propex_rebase_canada  if country==3
replace propex = propex * ppp_nz / ppp_aust     if country==2
replace propex = propex * ppp_canada / ppp_aust if country==3

* Regression log variables
gen lvc = log(opex / propex)
gen ly1 = log(custnum)
gen ly2 = log(circlen)
gen ly3 = log(rmdem)
gen lz1 = log(shareugc)

* mean correct output data
qui reg ly1
predict mly1
replace ly1=ly1-mly1
qui reg ly2
predict mly2
replace ly2=ly2-mly2
qui reg ly3
predict mly3
replace ly3=ly3-mly3

* TLG variables
gen ly11 = (ly1^2)/2
gen ly12 = ly1*ly2
gen ly13 = ly1*ly3
gen ly22 = (ly2^2)/2
gen ly23 = ly2*ly3
gen ly33 = (ly3^2)/2

*/  Time Trend Variables
gen aus_yr = cond(country == 1,yr,0)
gen nz_yr = cond(country == 2,yr,0)
gen ont_yr = cond(country == 3,yr,0)

* Labels
label variable ly1  "ln(Custnum)=x1" 
label variable ly2  "ln(CircLen)=x2" 
label variable ly3  "ln(RMDemand)=x3" 
label variable lz1  "ln(ShareUGC)" 
label variable ly11 "x1*x1/2" 
label variable ly12 "x1*x2" 
label variable ly13 "x1*x3" 
label variable ly22 "x2*x2/2" 
label variable ly23 "x2*x3" 
label variable ly33 "x3*x3/2"
label variable yr   "Year" 
label variable jur2 "New Zealand" 
label variable jur3 "Ontario" 

* Documentation

putdocx begin
putdocx paragraph, style(Title)
putdocx text ("Time Varying BC95-JTT-HN-`perd' period" )

* Regression models
*---------------------- 
	
*===============================================
* SFACD 
*================================================

constraint 1 _b[mu:_cons] == 0

qui reg lvc ly1 ly2 ly3 lz1 yr jur2 jur3
matrix b0 = e(b), ln(e(rmse)^2) , .1, .1

qui xtfrontier lvc ly1 ly2 ly3 lz1 yr jur2 jur3, cost from(b0, copy) ti ///
	diff constraints(1)
matrix b1 = e(b) 
matrix list b1
matrix b1f = b1[1,1..8]
matrix list b1f
	
matrix b2 = J(1, 3, 0)
matrix b2[1, 1] = 0.0001
matrix b2[1, 2] = 0.0001
matrix b2[1, 3] = 0.0001
matrix list b2 

sfpanel lvc ly1 ly2 ly3 lz1 yr jur2 jur3, model(bc95) ///
		emean(aus_yr nz_yr ont_yr, nocons) cost  svfrontier(b1f) svemean(b2) ///
		technique(dfp) iterate(500)

estimates store SFACDBC95

* STATS & TESTS

predict yhat1, xb
predict yhat2, u
gen yhat = yhat1 + yhat2	
gen eh = lvc - yhat
predict bc, bc ci

estat ic

scalar k_sfacd = 13 		
qui corr lvc yhat
local psadjr2 = 1 - ((1 - r(rho)^2)*(r(N) - 1)/(r(N) - k_sfacd))	
display "SFACD Pseudo-adjusted R2 = " `psadjr2'

iqr eh
swilk eh
collin ly1 ly2 ly3 lz1 yr jur2 jur3
linktest

* OUTPUT ELASTICITIES	
predictnl ely1 = (_b[ly1]), ci(lw_y1 up_y1)
predictnl ely2 = (_b[ly2]), ci(lw_y2 up_y2)
predictnl ely3 = (_b[ly3]), ci(lw_y3 up_y3)
predictnl elY  = (_b[ly1] + _b[ly2] + _b[ly3]), ci(lw_Y up_Y)

label var ely1 "ely1"
label var ely2 "ely2"
label var ely3 "ely3"
label var elY  "elY"
label var lw_y1 "lw_y1"
label var up_y1 "up_y1"
label var lw_y2  "lw_y2"
label var up_y2  "up_y2"
label var lw_y3  "lw_y3"
label var up_y3  "up_y3"
label var lw_Y  "lw_Y"
label var up_Y  "up_Y"

putdocx paragraph, style(Subtitle)
putdocx text ("SFACD BC95-JTT-HN Elasticities")
table (country) (var) , stat(mean ely1 ely2 ely3 elY) nformat(%5.3f)
putdocx collect

*EFFICIENCY SCORES
 
export excel eiid year bc bc_LB95 bc_UB95 using ///
	"Output/xlsx/SFA-BC95-JTT-HN/SFA-BC95-JTT-HN-eff-`perd'.xlsx" if eiid < 2000, ///
    sheet("SFACD") firstrow(var) replace nolabel

putdocx paragraph, style(Subtitle)
putdocx text ("SFACD BC95-JTT-HN Efficiency Scores - `perd' period")
table (country) (var), stat(mean bc bc_LB95 bc_UB95) nformat(%5.3f)
putdocx collect
table (dnsp) (var), stat(mean bc bc_LB95 bc_UB95) nformat(%5.3f)
putdocx collect

*Charts	
twoway (line bc year if dnsp == 1,  sort lcolor(blue)      lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 2,  sort lcolor(red)       lpattern(dash)        lwidth(0.5)) ///
       (line bc year if dnsp == 3,  sort lcolor(green)     lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 4,  sort lcolor(orange)    lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 5,  sort lcolor(maroon)    lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 6,  sort lcolor(teal)      lpattern(dash)        lwidth(0.5)) ///
       (line bc year if dnsp == 7,  sort lcolor(magenta)   lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 8,  sort lcolor(dkgreen)   lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 9,  sort lcolor(purple)    lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 10, sort lcolor(brown)     lpattern(dash)        lwidth(0.5)) ///
       (line bc year if dnsp == 11, sort lcolor(navy)      lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 12, sort lcolor(pink)      lpattern(solid) 	     lwidth(0.5)) ///
       (line bc year if dnsp == 13, sort lcolor(olive)     lpattern(solid)       lwidth(0.5)), ///
       legend(order(1 "EVO" 2 "AGD" 3 "CIT" 4 "END" 5 "ENX" ///
                    6 "ERG" 7 "ESS" 8 "JEN" 9 "PCR" 10 "SAP" ///
                    11 "AND" 12 "TND" 13 "UED") title("DNSP")) ///
       xtitle("Year") ytitle("Efficiency Score") ///
       xlabel(`startyr'(1)`endyr', angle(90) grid)  /// 
       ylabel(0(0.1)1, grid) /// 
       xscale(range(`startyr' `endyr')) /// 
       yscale(range(0 1))       

graph export "Output/chart/SFA-BC95-JTT-HN/SFACD-BC95-JTT-HN-Eff-`perd'.png", ///
	replace
putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFACD BC95-JTT-HN Efficiency Scores - `perd' period")	
putdocx paragraph, halign(center)
putdocx image "Output/chart/SFA-BC95-JTT-HN/SFACD-BC95-JTT-HN-Eff-`perd'.png", ///
	width(8) height(5.6)
 
scatter eh yhat if country == 1, mcolor(blue) ///
    || scatter eh yhat if country == 2, mcolor(red) ///
    || scatter eh yhat if country == 3, mcolor(green) ///
    yline(0) ///
    ytitle("Residuals") ///
    xtitle("Fitted Values") ///
    legend(order(1 "AUS" 2 "NZ" 3 "ONT"))
graph export "Output/chart/SFA-BC95-JTT-HN/SFACD-BC95-JTT-HN-Res-`perd'.png", ///
	replace
putdocx paragraph, halign(center)
putdocx image "Output/chart/SFA-BC95-JTT-HN/SFACD-BC95-JTT-HN-Res-`perd'.png", ///
	width(8) height(5.6)
 

estout  SFACDBC95 using "Output/xlsx/SFA-BC95-JTT-HN/SFACD-BC95-JTT-HN-Reg-`perd'.tsv", ///
	cells("b(fmt(4)) se(fmt(4)) t(fmt(4))") style(tab) label ///
	title("SFACD-BC95-JTT-HN-`perd'") stats(N bic, fmt(3)) replace 
  
drop bc bc_LB95 bc_UB95 ely1 lw_y1 up_y1 ely2 lw_y2 up_y2 ely3 lw_y3 up_y3 ///
	   elY lw_Y up_Y yhat1 yhat2 yhat eh
	   
*===================================================
*  SFATLG 
*===================================================

constraint 1 _b[mu:_cons] == 0

qui reg lvc ly1 ly2 ly3 ly11 ly12 ly13 ly22 ly23 ly33 lz1 yr jur2 jur3
matrix b0 = e(b), ln(e(rmse)^2) , .1, .1

qui xtfrontier lvc ly1 ly2 ly3 ly11 ly12 ly13 ly22 ly23 ly33 lz1 yr jur2 jur3, ///
		cost from(b0, copy) ti diff constraints(1)

matrix b1 = e(b) 
matrix list b1
matrix b1f = b1[1,1..14]
matrix list b1f

matrix b2 = J(1, 3, 0)
matrix b2[1, 1] = 0.0001
matrix b2[1, 2] = 0.0001
matrix b2[1, 3] = 0.0001
matrix list b2 

sfpanel lvc ly1 ly2 ly3 ly11 ly12 ly13 ly22 ly23 ly33 lz1 yr jur2 jur3, ///
	model(bc95) emean(aus_yr nz_yr ont_yr, nocons) cost  svfrontier(b1f) ///
	svemean(b2) technique(dfp) iterate(500)

estimates store SFATLGBC95

* STATS & TESTS

predict yhat1, xb
predict yhat2, u
gen yhat = yhat1 + yhat2	
gen eh = lvc - yhat	
predict bc, bc ci 

estat ic

scalar k_sfatlg = 19 		
qui corr lvc yhat
local psadjr2 = 1 - ((1 - r(rho)^2)*(r(N) - 1)/(r(N) - k_sfacd))	
display "SFATLG Pseudo-adjusted R2 = " `psadjr2'

iqr eh
swilk eh
collin ly1 ly2 ly3 ly11 ly12 ly13 ly22 ly23 ly33 lz1 yr jur2 jur3
linktest

test ly11 ly12 ly13 ly22 ly23 ly33

* OUTPUT ELASTICITIES	
	
predictnl ely1 = (_b[ly1] + _b[ly11]*ly1 + _b[ly12]*ly2 + _b[ly13]*ly3), ///
	ci(lw_y1 up_y1) p(p1)
predictnl ely2 = (_b[ly2] + _b[ly12]*ly1 + _b[ly22]*ly2 + _b[ly23]*ly3), ///
	ci(lw_y2 up_y2) p(p2)
predictnl ely3 = (_b[ly3] + _b[ly13]*ly1 + _b[ly23]*ly2 + _b[ly33]*ly3), ///
	ci(lw_y3 up_y3) p(p3)
predictnl elY  = (_b[ly1]  + _b[ly11]*ly1 + _b[ly12]*ly2 + _b[ly13]*ly3 + ///
	_b[ly2] + _b[ly12]*ly1 + _b[ly22]*ly2 + _b[ly23]*ly3 + _b[ly3] + ///
	_b[ly13]*ly1 + _b[ly23]*ly2 + _b[ly33]*ly3), ci(lw_Y up_Y)

label var ely1 "ely1"
label var ely2 "ely2"
label var ely3 "ely3"
label var elY  "elY"
label var lw_y1 "lw_y1"
label var up_y1 "up_y1"
label var lw_y2  "lw_y2"
label var up_y2  "up_y2"
label var lw_y3  "lw_y3"
label var up_y3  "up_y3"
label var lw_Y  "lw_Y"
label var up_Y  "up_Y"

putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFATLG BC95-JTT-HN Elasticities - `perd' period")
table (country) (var) , stat(mean ely1 ely2 ely3 elY) nformat(%5.3f)
putdocx collect
table (dnsp) (var) , stat(mean ely1 ely2 ely3 elY) nformat(%5.3f)
putdocx collect

display "MONOTONICITY VIOLATIONS"
gen mon1 = 0
gen mon2 = 0
gen mon3 = 0
gen montot = 0
replace mon1 = cond(ely1 < 0,100,0)	
replace mon2 = cond(ely2 < 0,100,0)	
replace mon3 = cond(ely3 < 0,100,0)
replace montot = 100 if mon1 == 100 | mon2 == 100 | mon3 == 100

putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFATLG BC95-JTT-HN Monotonicity Violations - `perd' period")
table (country) (var) , stat(mean mon1 mon2 mon3 montot) nformat(%5.1f)
putdocx collect
table (dnsp) (var) , stat(mean mon1 mon2 mon3 montot) nformat(%5.1f)
putdocx collect

export excel eiid year country custnum rmdem circlen ely1 ely2 ely3 ///
	using "Output/xlsx/SFA-BC95-JTT-HN/monotonicity-`perd'.xlsx", ///
	sheet("SFATLG", replace) firstrow(var) 	

*EFFICIENCY SCORES
	
export excel eiid year bc bc_LB95 bc_UB95 using ///
	"Output/xlsx/SFA-BC95-JTT-HN/SFA-BC95-JTT-HN-eff-`perd'.xlsx" if eiid < 2000, ///
    sheet("SFATLG", modify) firstrow(var) nolabel

putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFATLG BC95-JTT-HN Efficiency Scores - `perd' period")
table (country) (var), stat(mean bc bc_LB95 bc_UB95) nformat(%5.3f)
putdocx collect
table (dnsp) (var), stat(mean bc bc_LB95 bc_UB95) nformat(%5.3f)
putdocx collect

twoway (line bc year if dnsp == 1,  sort lcolor(blue)      lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 2,  sort lcolor(red)       lpattern(dash)        lwidth(0.5)) ///
       (line bc year if dnsp == 3,  sort lcolor(green)     lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 4,  sort lcolor(orange)    lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 5,  sort lcolor(maroon)    lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 6,  sort lcolor(teal)      lpattern(dash)        lwidth(0.5)) ///
       (line bc year if dnsp == 7,  sort lcolor(magenta)   lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 8,  sort lcolor(dkgreen)   lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 9,  sort lcolor(purple)    lpattern(solid)       lwidth(0.5)) ///
       (line bc year if dnsp == 10, sort lcolor(brown)     lpattern(dash)        lwidth(0.5)) ///
       (line bc year if dnsp == 11, sort lcolor(navy)      lpattern(longdash)    lwidth(0.5)) ///
       (line bc year if dnsp == 12, sort lcolor(pink)      lpattern(solid) 	  	 lwidth(0.5)) ///
       (line bc year if dnsp == 13, sort lcolor(olive)     lpattern(solid)       lwidth(0.5)), ///
       legend(order(1 "EVO" 2 "AGD" 3 "CIT" 4 "END" 5 "ENX" ///
                    6 "ERG" 7 "ESS" 8 "JEN" 9 "PCR" 10 "SAP" ///
                    11 "AND" 12 "TND" 13 "UED") title("DNSP")) ///
       xtitle("Year") ytitle("Efficiency Score") ///
       xlabel(`startyr'(1)`endyr', angle(90) grid)  /// 
       ylabel(0(0.1)1, grid) /// 
       xscale(range(`startyr' `endyr')) /// 
       yscale(range(0 1))       
      	   
graph export "Output/chart/SFA-BC95-JTT-HN/SFATLG-BC95-JTT-HN-Eff-`perd'.png", ///
	replace
putdocx paragraph, halign(center)
putdocx image "Output/chart/SFA-BC95-JTT-HN/SFATLG-BC95-JTT-HN-Eff-`perd'.png", ///
	width(8) height(5.6)

* Residuals charts
	
scatter eh yhat if country == 1, mcolor(blue) ///
    || scatter eh yhat if country == 2, mcolor(red) ///
    || scatter eh yhat if country == 3, mcolor(green) ///
    yline(0) ///
    ytitle("Residuals") ///
    xtitle("Fitted Values") ///
    legend(order(1 "AUS" 2 "NZ" 3 "ONT"))
graph export "Output/chart/SFA-BC95-JTT-HN/SFATLG-BC95-JTT-HN-Res-`perd'.png", ///
	replace
putdocx paragraph, halign(center)
putdocx image "Output/chart/SFA-BC95-JTT-HN/SFATLG-BC95-JTT-HN-Res-`perd'.png", ///
	width(8) height(5.6)

estout  SFATLGBC95 using "Output/xlsx/SFA-BC95-JTT-HN/SFATLG-BC95-JTT-HN-Reg-`perd'.tsv", ///
	cells("b(fmt(4)) se(fmt(4)) t(fmt(4))") style(tab) label ///
	title("SFATLG-BC95-JTT-HN`perd'") stats(N bic, fmt(3)) replace	
	
*-----------------------------------------------------	
* ALTERNATIVE OUTPUT ELASTICITIES	
*-----------------------------------------------------

if "`perd'" == "long" {		
	predictnl aely1 = (_b[ly1] + _b[ly13]*ly3), ci(alw_y1 aup_y1)
	predictnl aely2 = (_b[ly2]), ci(alw_y2 aup_y2)
	predictnl aely3 = (_b[ly3] + _b[ly13]*ly1 + _b[ly33]*ly3), ci(alw_y3 aup_y3)
	predictnl aelY  = (_b[ly1]  + _b[ly13]*ly3 + _b[ly2] + _b[ly3] + ///
		_b[ly13]*ly1 + _b[ly33]*ly3), ci(alw_Y aup_Y)
}
else if "`perd'" == "short" {
	predictnl aely1 = (_b[ly1] + _b[ly13]*ly3), ci(alw_y1 aup_y1)
	predictnl aely2 = (_b[ly2]), ci(alw_y2 aup_y2)
	predictnl aely3 = (_b[ly3] + _b[ly13]*ly1 + _b[ly33]*ly3), ci(alw_y3 aup_y3)
	predictnl aelY  = (_b[ly1]  + _b[ly13]*ly3 + _b[ly2] + _b[ly3] + ///
		_b[ly13]*ly1 + _b[ly33]*ly3), ci(alw_Y aup_Y)	
}		
	
label var aely1 "aely1"
label var aely2 "aely2"
label var aely3 "aely3"
label var aelY  "aelY"	
	
putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFATLG BC95-JTT-HN Alternative Elasticities - `perd' period")
table (country) (var) , stat(mean aely1 aely2 aely3 aelY) nformat(%5.3f)
putdocx collect
table (dnsp) (var) , stat(mean aely1 aely2 aely3 aelY) nformat(%5.3f)
putdocx collect

display "ALTERNATIVE MONOTONICITY VIOLATIONS"
gen mv1 = 0
gen mv2 = 0
gen mv3 = 0
gen mvtot = 0
replace mv1 = cond(aely1 < 0,100,0)	
replace mv2 = cond(aely2 < 0,100,0)	
replace mv3 = cond(aely3 < 0,100,0)
replace mvtot = 100 if mv1 == 100 | mv2 == 100 | mv3 == 100

putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFATLG BC95-JTT-HN Alternative Monotonicity Violations - `perd' period")
table (country) (var) , stat(mean mv1 mv2 mv3 mvtot) nformat(%5.1f)
putdocx collect
table (dnsp) (var) , stat(mean mv1 mv2 mv3 mvtot) nformat(%5.1f)
putdocx collect		
	
*============================================================
  
putdocx save "Output/docx/SFA-BC95-JTT-HN/BC95-JTT-HN-`perd'", replace

**
log close
exit
